using System;
using System.Xml;
using System.Reflection;
using System.Collections;
using System.Diagnostics;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Http;

namespace Team_Project.PersistencyManagers.Protocols
{
	/// <summary>
	/// Enumerativo che indica la direzione per la quale verr utilizzato un protocollo
	/// </summary>
	public enum Direction
	{
		/// <summary>
		/// Direzione di scrittura o cancellazione
		/// </summary>
		Store,
		/// <summary>
		/// Direzione di lettura
		/// </summary>
		Retrive
	}

	/// <summary>
	/// Delegato utilizzato per la chiamata asincrona dell'operazione di richiesta di
	/// un protocollo
	/// </summary>
	public delegate IPersistencyProtocol GetProtocolDelegate(string projectName, string asmName, string elTypeName, Direction forDirection);

	/// <summary>
	/// Gestore dei protocolli. Questa classe permette di accedere al protocollo
	/// corretto per l'utilizzo di un certo tipo di elemento.
	/// </summary>
	/// <remarks>L'istanza globale di questa classe  disponibile con il
	/// nome "ProtocolManager"</remarks>
	[TeamProjectInit("ProtocolManager")]
	public class ProtocolManager: MarshalByRefObject
	{
		private string configPath;
		private bool useProjectName;
		private Hashtable protocols = new Hashtable();
		private ObjRef lrmReg;

		/// <summary>
		/// Crea un protocolManager inizializzato.
		/// <note>Questa funzione  utilizzata dall'engine all'avvio del servizio</note>
		/// </summary>
		/// <returns>Un protocol Manager inizializzato</returns>
		public static ProtocolManager ComponentInit()
		{
			return new ProtocolManager(@"..\..\..\Config\Protocols\",true);
		}

		/// <summary>
		/// Costruisce un protocol manager utilizzando come file di configurazione
		/// Config\Protocols\Protocols.xml
		/// </summary>
		public ProtocolManager():
#if DEBUG
			this(@"..\..\..\Config\Protocols\Protocols.xml",false)
#else
			this(@"Config\Protocols\Protocols.xml",false)
#endif
		{}

		/// <summary>
		/// Costruisce un manager di protocolli
		/// </summary>
		/// <param name="cfgBase">Percorso dove sono localizzati i file di configurazione
		/// se il parametro useProjectAsFileName  true, altrimenti il nome del
		/// file di configurazione (con percorso assoluto o relativo)</param>
		/// <param name="useProjectAsFileName">true se si desidera file di configurazione
		/// diversi per ogni protocollo, false se si desidera utilizzarne uno unico.</param>
		public ProtocolManager(string cfgBase,bool useProjectAsFileName)
		{
			useProjectName = useProjectAsFileName;
			if(useProjectName && cfgBase != string.Empty && !cfgBase.EndsWith(@"\"))
				this.configPath = cfgBase + @"\";
			else
				this.configPath = cfgBase;
			IDictionary props = new Hashtable();
			props.Add("typeFilterLevel","Full");
			SoapServerFormatterSinkProvider fsp = new SoapServerFormatterSinkProvider(props,null);
			IDictionary Chprops = new Hashtable();
			int prtstr = Globals.Instance.ListenPort;
			Chprops.Add("port",prtstr.ToString());
			Chprops.Add("name","http");
			HttpChannel channel = new HttpChannel(Chprops,new SoapClientFormatterSinkProvider(),fsp);
			ChannelServices.RegisterChannel(channel);
			Trace.WriteLine(channel.ChannelName + " registred");// at port " + channel.Properties["port"].ToString());

			lrmReg = RemotingServices.Marshal(this, "PersistencyManager.rem");
			Trace.WriteLine("OBJ: "+lrmReg.URI);
		}

		/// <summary>
		/// Restituisce il protocollo da utilizzare per accedere ad un elemento.
		/// </summary>
		/// <param name="projectName">Nome del progetto al quale appartiene la richiesta</param>
		/// <param name="t">Tipo dell'elemento al quale si desidera accedere</param>
		/// <param name="forDirection">Direzione di accesso all'elemento</param>
		/// <returns>Il protocollo opportuno sul quale effettuare le richieste</returns>
		public IPersistencyProtocol GetProtocolFor(string projectName, Type t, Direction forDirection)
		{
			return GetProtocolFor(projectName,t.Assembly.FullName,t.FullName,forDirection);
		}

		/// <summary>
		/// Restituisce il protocollo da utilizzare per accedere ad un elemento.
		/// </summary>
		/// <remarks>Questa funzione  utilizzata per l'accesso sicuro in remoto.
		/// Essendo la classe System.Type difficilmente serializzabile  sufficiente
		/// inviare in remoto solo le stringhe necessarie ad identificare l'assembly
		/// e il tipo.</remarks>
		/// <param name="projectName">Nome del progetto al quale appartiene la richiesta</param>
		/// <param name="asmName">Nome dell'assembly dove  contenuto il tipo di
		/// elemento da utilizzare</param>
		/// <param name="elTypeName">Nome del tipo dell'elemento che si desidera utilizzare</param>
		/// <param name="forDirection">Direzione di accesso all'elemento</param>
		/// <returns>Il protocollo opportuno sul quale effettuare le richieste</returns>
		public IPersistencyProtocol GetProtocolFor(string projectName, string asmName, string elTypeName, Direction forDirection)
		{
			string key = projectName + "##" + asmName + "##" + elTypeName + "##" + forDirection.ToString();
			IPersistencyProtocol pp = (IPersistencyProtocol)protocols[key];
			if(pp == null)
			{
				Assembly asm = Assembly.Load(asmName);
				Type ElementType = asm.GetType(elTypeName);
				XmlDocument doc = new XmlDocument();
				string cfgN = configPath;
				if(useProjectName) cfgN += projectName + ".xml";
				doc.Load(cfgN);
				string xpath = "//Protocols/Project[@name=\""+ projectName+	"\"]/Element[@name=\""
					+ElementType.Name+"\"]/"+forDirection.ToString()+ "Protocol";
			
				Trace.WriteLine("GetProtocolFor-XPath: " + xpath);
			
				XmlNode elem = doc.SelectSingleNode(xpath);
				if(elem != null)
				{
					Assembly a = Assembly.Load(elem.Attributes["assembly"].Value);
					Type t = a.GetType(elem.Attributes["type"].Value);
					Trace.WriteLine("Trying to create a " + t.Name + " with params " + projectName
						+ ", " + ElementType.Name /*+ ", " + elem.OuterXml*/);
					pp = (IPersistencyProtocol)Activator.CreateInstance(t, new object[]{projectName,ElementType,elem});
					protocols[key] = pp;
					pp.Intit();
				}
			}
			return pp;
		}
	}
}
